/**
* \file: gentexgstbuf.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* Implementation of generating texture and gst buffer
*
* \component: gentexgstbuf
*
* \author: Jens Georg <jgeorg@de.adit-jv.com> , Anil V <anil.valmiki@in.bosch.com>
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/

extern "C" {
#define GL_GLEXT_PROTOTYPES
}
#include "GLES2/gl2.h"
#include <GLES2/gl2ext.h>
#include "gentexgstbuf.h"
#if !GST_CHECK_VERSION(1,0,0)
#include <gst/fsl/gstbufmeta.h>
#else
#include <EGL/egl.h>
#include <EGL/eglext.h>
extern "C" {
#define virtual virt
#include <drm/drm.h>
#undef virtual
#include <drm/drm_fourcc.h>
}
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <xf86drm.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <gst/allocators/gstdmabuf.h>
#include <gst/video/video-frame.h>
#endif
#include <memory>
#include "memorypool.h"
#include <sys/time.h>

static const std::string KEY_POOL_SIZE = "PoolSize";
static uint minTextureSize = 64;
static uint maxTextureSize = 8192;

/* right now WITH_DMABUF not works, needs to check */
#undef WITH_DMABUF

namespace generateTexGstbuf {

/**
 * @brief Helper texture class
 * Used to be able to attach textures
 */
class Texture
{
public:
    uint _width;
    uint _height;
    GLuint _id;
#if !GST_CHECK_VERSION(1,0,0)
    GstBufferMeta *get_meta();
    void set_meta(GstBufferMeta *);
#else
    uint32_t _stride;
    uint32_t _handle;
    int _dma_fdesc;
    EGLImageKHR _eglImage;
    EGLDisplay _egldpy;
    uint32_t GetDrmBufferFormat(GLint textureformat);
    GstAllocator *get_gstdmabuf();
    void set_gstdmabuf(GstAllocator *);
#endif
    void get_tex_info(void **, void**);
    Texture(uint width, uint height, GLuint textureFormat, void *obj);
    Texture():_width(512),_height(512),_id(0),_log_data(NULL),_phy_data(NULL)
    {}
    ~Texture();
private:
    void *_log_data;
    void *_phy_data;
#if !GST_CHECK_VERSION(1,0,0)
    GstBufferMeta *_meta;
#else
    GstAllocator *_gstdmabuf;
#endif
    void set_tex_info(void *, void*);
};

class Pool_alloc: public Texture
{
public:
    /**
      * @brief Helper struct to allocate and destroy GenTexGstbuf::Texture in Tex
      * pool
      */
     struct TexAllocator : public Allocator<Texture>
     {
         virtual Texture *alloc(void *);
         virtual void destroy(Texture *tex);
     };
     // Template realisation of pool for Textures
     typedef Pool<Texture, TexAllocator> TexPool;

     std::auto_ptr<TexPool> _texPOOL;

     Pool_alloc(void *obj)
     {
         _texPOOL = std::auto_ptr<TexPool>(new TexPool(obj, 5, TexPool::INITIALLY_EMPTY));
     }

     void Pool_reset(void *obj,int poolSize)
     {
         _texPOOL.reset(new TexPool(obj, poolSize, TexPool::INITIALLY_EMPTY));
     }

     void Pool_reset()
     {
         _texPOOL.reset();
     }
};

Texture *Pool_alloc::TexAllocator::alloc(void *obj)
{
    Texture *texture ;
    GenTexGstbuf *_obj = (GenTexGstbuf *)obj;
    texture = new Texture(_obj->getTexWidth(), _obj->getTexHeight(), _obj->getTextureFormat(), (void *)_obj);
    _obj->tex_map.insert( std::pair<int,Texture *>(texture->_id,texture));
    return texture;
}

void Pool_alloc::TexAllocator::destroy(Texture *texture)
{
    delete texture;
}

void Texture::get_tex_info(void** log_data, void** phy_data)
{
    *log_data = _log_data;
    *phy_data = _phy_data;
}
void Texture::set_tex_info(void* log_data, void* phy_data)
{
    _log_data = log_data;
    _phy_data = phy_data;
}

#if !GST_CHECK_VERSION(1,0,0)
GstBufferMeta *Texture::get_meta()
{
    return _meta;
}

void Texture::set_meta(GstBufferMeta *meta)
{
    _meta = meta;
}
#else

GstAllocator *Texture::get_gstdmabuf()
{
    return _gstdmabuf;
}

void Texture::set_gstdmabuf(GstAllocator *gstdmabuf)
{
    _gstdmabuf = gstdmabuf;
}

uint32_t Texture::GetDrmBufferFormat(GLint textureFormat)
{
    uint32_t drm_buf_format = 0;
    switch (textureFormat)
    {
    case GL_RGB565:
        drm_buf_format = DRM_FORMAT_RGB565;
        break;
    case GL_RGBA:
        drm_buf_format = DRM_FORMAT_ABGR8888;
        break;
    default:
        printf("Default case: %d\n", textureFormat);
    }
    return drm_buf_format;
}
#endif

Texture::Texture(uint width, uint height, GLuint textureFormat, void *obj)
    :_width(width)
    ,_height(height)
    ,_id(0)
    ,_log_data(NULL)
    ,_phy_data(NULL)
{
#if GPU_VIVANTE

    void *l;
    void *p;

    set_meta(NULL);
    glGenTextures(1, &_id);
    glBindTexture(GL_TEXTURE_2D, _id);
    glTexDirectPhysVIV(GL_TEXTURE_2D, _width, _height, textureFormat, (GLvoid **) &l, (GLuint *) &p);
    glTexDirectInvalidateVIV(GL_TEXTURE_2D);

    set_tex_info(l,p);
#else
    _stride = 0;
    _handle = 0;
    _dma_fdesc = 0;
    _eglImage = EGL_NO_IMAGE_KHR;
    _egldpy = NULL;
    set_gstdmabuf(NULL);


    int fd;
    int dma_fdesc;
    EGLint attribs[30] = { 0 };
    struct drm_mode_create_dumb create_arg = {0};
    struct drm_prime_handle fd_arg = {0};
    struct drm_mode_map_dumb map_arg = {0};
    int atti = 0;

    int ret = 0;

    void *l;
    void *p;

    GenTexGstbuf *_obj = (GenTexGstbuf *)obj;
    fd = _obj->getDrm_fd();

    /*allocate the buffer*/
    errno = 0;
    create_arg.bpp = 32;
    create_arg.width = width;
    create_arg.height = height;
    ret = drmIoctl(fd, DRM_IOCTL_MODE_CREATE_DUMB, &create_arg);
    if (ret) {
        std::cout<<"drm create dump failed, errmsg "<<strerror(errno)<<std::endl;
    }

    /*getting dma fd*/
    _width = width;
    _height = height;
    _stride = create_arg.pitch;
    _handle = create_arg.handle;
    map_arg.handle = _handle;
    ret = drmIoctl(fd, DRM_IOCTL_MODE_MAP_DUMB, &map_arg);

    l = (void*)mmap(0, (_height * _stride), PROT_READ | PROT_WRITE, \
                                MAP_SHARED, fd, map_arg.offset);

    perror("mmap");

    if(l == MAP_FAILED)
    {
        std::cout<<"mmap failed, errmsg "<<strerror(errno)<<std::endl;
    }
    /*We cannot get physical address for the dmafd*/
    p = 0;

    errno = 0;
    fd_arg.fd = 0;
    fd_arg.handle = _handle;
    ret = drmIoctl(fd, DRM_IOCTL_PRIME_HANDLE_TO_FD, &fd_arg);

    if ((ret < 0) || (fd_arg.fd < 0)) {
        struct drm_mode_destroy_dumb destroy_arg = {0};

        std::cout<<"get dma fd failed, errmsg "<<strerror(errno)<<std::endl;
        destroy_arg.handle = fd_arg.handle;
        drmIoctl(fd, DRM_IOCTL_MODE_DESTROY_DUMB, &destroy_arg);
    }

    _dma_fdesc =  fd_arg.fd;

    glGetError();
    glGenTextures(1, &_id);
    glBindTexture(GL_TEXTURE_2D, _id);

    attribs[atti++] = EGL_WIDTH;
    attribs[atti++] = width;
    attribs[atti++] = EGL_HEIGHT;
    attribs[atti++] = height;
    attribs[atti++] = EGL_LINUX_DRM_FOURCC_EXT;
    attribs[atti++] = GetDrmBufferFormat(textureFormat);

    attribs[atti++] = EGL_DMA_BUF_PLANE0_FD_EXT;
    attribs[atti++] = _dma_fdesc;
    attribs[atti++] = EGL_DMA_BUF_PLANE0_OFFSET_EXT;
    attribs[atti++] = 0;
    attribs[atti++] = EGL_DMA_BUF_PLANE0_PITCH_EXT;
    attribs[atti++] = _stride;
    attribs[atti++] = EGL_NONE;

    _egldpy = eglGetCurrentDisplay();
    if(EGL_NO_DISPLAY == _egldpy){
        printf("Failed to obtain current display: %x \n", eglGetError());
    }

    PFNEGLCREATEIMAGEKHRPROC eglCreateImageKHR =
            (PFNEGLCREATEIMAGEKHRPROC) eglGetProcAddress("eglCreateImageKHR");
    _eglImage = eglCreateImageKHR(_egldpy, EGL_NO_CONTEXT, EGL_LINUX_DMA_BUF_EXT, NULL , attribs);
    if (EGL_NO_IMAGE_KHR == _eglImage)
    {
        printf("Failed to create egl image from dma buffer : %x \n", eglGetError());
    }

    PFNGLEGLIMAGETARGETTEXTURE2DOESPROC glEGLImageTargetTexture2DOES =
            (PFNGLEGLIMAGETARGETTEXTURE2DOESPROC) eglGetProcAddress("glEGLImageTargetTexture2DOES");

    if (glEGLImageTargetTexture2DOES)
    glEGLImageTargetTexture2DOES(GL_TEXTURE_2D, _eglImage);
    else
    printf("glEGLImageTargetTexture2DOES is null \n");

    set_tex_info(l,p);
#endif
}

Texture::~Texture()
{
#if GST_CHECK_VERSION(1,0,0)
    PFNEGLDESTROYIMAGEKHRPROC  eglDestroyImageKHR = (PFNEGLDESTROYIMAGEKHRPROC) eglGetProcAddress("eglDestroyImageKHR");
    eglDestroyImageKHR(_egldpy, _eglImage);

    GstAllocator *_gstdmabuf = get_gstdmabuf();
    if(_gstdmabuf != NULL)
    {
        gst_object_unref(_gstdmabuf);
    }

    munmap(0, (_height * _stride));

    if(_dma_fdesc)
        close(_dma_fdesc);
#endif

    if(_id)
        glDeleteTextures(1, &_id);
}

/**
 * @brief GenTexGstbuf constructor set Buffertraits and
 * set texture pool size
 * @param BufferTraits
 */
GenTexGstbuf::GenTexGstbuf(const BufferTraits &bufTraits)
    :texHeight(0),
     texWidth(0),
     _bufTraits(),
     p_alloc(NULL),
     _textureFormat(0),
     tex_map(),
     check_size(true)
{
    p_alloc = new Pool_alloc(this);

    if(!p_alloc)
        std::cout << "failed to create pool"<< std::endl;

    setBufferTraits(bufTraits);

    BufferTraits::Config::const_iterator it = bufTraits.config.find(KEY_POOL_SIZE);
    if (it != bufTraits.config.end())
    {
        std::stringstream oss(it->second.c_str());
        int poolSize;
        oss >> poolSize;
        if (poolSize > 0)
        {
            p_alloc->Pool_reset(this,poolSize);
        }
    }
#if GST_CHECK_VERSION(1,0,0)
    errno = 0;
    int fd = 0;

    fd = open("/dev/dri/card0", O_RDWR);
    if (fd < 0)
    {
        std::cout<<"Opening drm dev node failed, errmsg "<<strerror(errno)<<std::endl;
    }

    setDrm_fd(fd);
#endif
}

GenTexGstbuf::~GenTexGstbuf()
{
    p_alloc->Pool_reset();
    delete p_alloc;
    p_alloc = NULL;

#if GST_CHECK_VERSION(1,0,0)
    int fd = getDrm_fd();
    if (0 != fd)
    {
        close(fd);
    }
#endif

    tex_map.clear();
}

/**
 * @brief helper function to align a value to the next multiple of 16
 * @param d value to alinge
 * @return m being a multiple of 16 for which d <= m
 */
static inline uint align16(uint d)
{
    return d % 16 == 0 ? d : d + (16 - (d % 16));
}

/**
 * @brief restrict a value to upper and lower bounds
 * @param val the initial value
 * @param min the lower bound
 * @param max the upper bound
 * @return val' with min <= val' <= max
 */
template <typename T>
static inline T clamp(const T &val, const T &min, const T &max)
{
    return std::max(min, std::min(max, val));
}


/**
 * @brief set the size of the Texture
 * @param width new width, height new height
 * @note Width will be aligned to be a multiple of 16 in the range of
 * [32,1920]
 * @note Height will be aligned to be in the range of [32,1080]
 */
void GenTexGstbuf::setTexSize(uint width, uint height)
{
    if(0 != texWidth && 0 != texHeight)
    {
        if(texWidth != width || texHeight != height)
        {
            /* clear existing textures in pool */
            p_alloc->_texPOOL->clear();
        }
    }

    texWidth = align16(clamp<uint>(width, minTextureSize, maxTextureSize));
    texHeight = align16(clamp<uint>(height, minTextureSize, maxTextureSize));

    check_size = true;
}

/**
 * @brief get the size of the Texture
 * @return width new width ,height new height after alignment
 */
void GenTexGstbuf::getTexSize(uint *width, uint *height)
{
    *width = texWidth;
    *height = texHeight;
}

/**
 * @brief nextTex get texture from the pool
 * @param NULL
 */
int GenTexGstbuf::nextTex(int wait_time)
{
    if(check_size)
    {
        glGetIntegerv(GL_MAX_TEXTURE_SIZE, (GLint*)&maxTextureSize);
        check_texture_size();
        check_size = false;
    }

    int rc;
    pthread_mutex_lock(&p_alloc->_texPOOL->sync_lock);
    if (p_alloc->_texPOOL->_active.size() == p_alloc->_texPOOL->_size)
    {
        if(wait_time == -1)
        {
            std::cout << "we reached max textures creation,"
                      <<"wait for infinite time for used texture to keep back in pool" << std::endl;
            // wait for textures in use to release
            p_alloc->_texPOOL->wait_free_texture = true;
            pthread_cond_wait(&p_alloc->_texPOOL->sync_cond, &p_alloc->_texPOOL->sync_lock);
        }
        else
        {
            struct timespec   ts;
            struct timeval    tp;
            rc =  gettimeofday(&tp, NULL);

            /* Convert from timeval to timespec */
                ts.tv_sec  = tp.tv_sec;
                ts.tv_nsec = tp.tv_usec * 1000;
                ts.tv_sec += wait_time;

            // wait for textures in use to release
            p_alloc->_texPOOL->wait_free_texture = true;
            rc = pthread_cond_timedwait(&p_alloc->_texPOOL->sync_cond,
                                        &p_alloc->_texPOOL->sync_lock,
                                        &ts);
        }
    }
    pthread_mutex_unlock(&p_alloc->_texPOOL->sync_lock);

    if (rc == ETIMEDOUT) {
//              return rc;
        std::cout << " wait time out "<<std::endl;
        return 0;
    }


    if(p_alloc->_texPOOL->pool_deleted)
    {
        std::cout << " In fun "<<__func__<<" pool is deleted "<<std::endl;
        return 0;
    }

    // get a free Texture from pool
    Texture *newtex = p_alloc->_texPOOL->get(this);

    if (NULL == newtex) {
        throw std::exception();
    }

    return newtex->_id;
}

/**
 * @brief releaseTex put texture back to pool,
 * Release meta data of gst buffer bind to a texture
 * @param NULL
 */
void GenTexGstbuf::releaseTex(uint tex_id)
{
    std::map<int,Texture *>::iterator it;

    it = tex_map.find(tex_id);
    if (it != tex_map.end())
    {
        Texture *tex = it->second;

        if(tex->_width != getTexWidth() || tex->_height != getTexHeight())
        {
            p_alloc->_texPOOL->steal(tex);
            tex_map.erase (it);
        }
        else
        {
            p_alloc->_texPOOL->put(tex);
        }

#if GST_CHECK_VERSION(1,0,0)
#if defined(WITH_DMABUF)
        GstAllocator *_gstdmabuf = tex->get_gstdmabuf();
        if(_gstdmabuf != NULL)
        {
            gst_object_unref(_gstdmabuf);
        }
        tex->set_gstdmabuf(NULL)
#endif
#else
        GstBufferMeta *meta = tex->get_meta();
        if(meta != NULL) {
            gst_buffer_meta_free(meta);
            tex->set_meta(NULL);
        }
#endif
    }
    else
    {
        std::cout << "fun "<<__func__<<" texture id not found "<<std::endl;
    }
}

/**
 * @brief getGstBufferFromTex get gst buffer from texture
 * @param GenTexGstbuf::Texture
 */
GstBuffer *GenTexGstbuf::getGstBufferFromTex(uint tex_id)
{
    std::map<int,Texture *>::iterator it;

    it = tex_map.find(tex_id);
    if (it == tex_map.end())
    {
        std::cout << "fun "<<__func__<<" texture id "<<tex_id<<"not found "<<std::endl;
        return NULL;
    }

    Texture *tex = it->second;

    // This can only be possible if the Tex was modified externally or not ours
    if (tex == NULL)
    {
        std::cout << "fun "<<__func__<<" invalid address "<<std::endl;
        return NULL;
    }

    void *l = NULL;
    void *p = NULL;
    tex->get_tex_info(&l,&p);

#if !GST_CHECK_VERSION(1,0,0)

    GstBuffer *buffer = gst_buffer_new();
    GST_BUFFER_DATA(buffer) = reinterpret_cast<guint8 *>(l);
    GST_BUFFER_SIZE(buffer) = getBufferSize();

    // Freescale magic! - Mark the buffer as a hardware buffer, this is
    // probably not needed but be sure.
    GstBufferMeta *meta;
    int index = G_N_ELEMENTS (buffer->_gst_reserved) - 1;
    meta = gst_buffer_meta_new ();
    meta->physical_data = (gpointer) (p);
    buffer->_gst_reserved[index] = meta;
    tex->set_meta(meta);

#else

    gsize offset[GST_VIDEO_MAX_PLANES] = {0, 0, 0, 0};
    gint stride[GST_VIDEO_MAX_PLANES] = {tex->_stride, 0, 0, 0};


#if defined(WITH_DMABUF)

    GstAllocator *_gstdmabuf = tex->get_gstdmabuf();
    if(_gstdmabuf != NULL)
    {
        gst_object_unref(_gstdmabuf);
    }

    GstBuffer *buffer = gst_buffer_new();
    _gstdmabuf = gst_dmabuf_allocator_new();

    GstMemory* gstmem = gst_dmabuf_allocator_alloc ((GstAllocator*)_gstdmabuf, tex->_dma_fdesc, getTexHeight()*tex->_stride);

    gst_buffer_append_memory(buffer, gstmem);

    gst_buffer_add_video_meta_full (
            buffer, GST_VIDEO_FRAME_FLAG_NONE,
            (GstVideoFormat)GST_VIDEO_FORMAT_RGB16,
            getTexWidth(),
            getTexHeight(),
            1,
            offset,
            stride);

    tex->set_gstdmabuf(_gstdmabuf);
#else
    GstBuffer *buffer = gst_buffer_new_wrapped_full( (GstMemoryFlags)0, reinterpret_cast<gpointer>(l),
                                                    getBufferSize(), 0, getBufferSize(), NULL, NULL );

    gst_buffer_add_video_meta_full (
            buffer, GST_VIDEO_FRAME_FLAG_NONE,
            (GstVideoFormat)GST_VIDEO_FORMAT_RGB16,
            getTexWidth(),
            getTexHeight(),
            1,
            offset,
            stride);
#endif
#endif

    return buffer;
}

void GenTexGstbuf::setBufferTraits(const BufferTraits &traits)
{
    BufferTraits currentTraits = traits;
    GLuint textureFormat = GL_RGBA;

    if (currentTraits.alpha == 0)
    {
        if (currentTraits.red == 5 && currentTraits.green == 6 && currentTraits.blue == 5)
        {
            textureFormat = GL_RGB565;
        }
        else
        {
            textureFormat = GL_RGBA;
            currentTraits.alpha = currentTraits.red = currentTraits.green = currentTraits.blue = 8;

            std::cout << "Unsupported traits, using fallback to GL_RGB";
        }
    }
    else if (currentTraits.alpha == 8)
    {
        if (currentTraits.red == 8 && currentTraits.green == 8 && currentTraits.blue == 8)
        {
            textureFormat = GL_RGBA;
        }
        else
        {
            textureFormat = GL_RGBA;
            currentTraits.red = currentTraits.green = currentTraits.blue = 8;

            std::cout << "Unsupported traits, using fallback to GL_RGBA";
        }
    }
    else
    {
        currentTraits.alpha = currentTraits.red = currentTraits.green = currentTraits.blue = 8;
        std::cout << "Unsupported traits, using fallback to GL_RGBA";
    }

    _textureFormat = textureFormat;
    _bufTraits = currentTraits;

}

#if GST_CHECK_VERSION(1,0,0)
int GenTexGstbuf::getDrm_fd()
{
    return drm_fd;
}

void GenTexGstbuf::setDrm_fd(int fd)
{
    drm_fd = fd;
}
#endif

uint GenTexGstbuf::getTexHeight()
{
    return texHeight;
}

uint GenTexGstbuf::getTexWidth()
{
    return texWidth;
}

uint GenTexGstbuf::getDepth()
{
    return _bufTraits.alpha + _bufTraits.red + _bufTraits.green +
           _bufTraits.blue;
}

GLuint GenTexGstbuf::getTextureFormat()
{
    return _textureFormat;
}

const BufferTraits &GenTexGstbuf::getBufferTraits() const
{
    return _bufTraits;
}

void GenTexGstbuf::check_texture_size()
{
    if(texWidth > maxTextureSize || texHeight > maxTextureSize )
    {
        setTexSize(texWidth, texHeight);

        std::cout << " User try to set texture size not compatible to gpu driver \n"
                  << " Setting texture size to "<<texWidth<<"x"<<texHeight<<std::endl;
    }
}
} // namespace generateTexGstbuf
